'use client';

import { useState, useEffect } from 'react';
import Link from 'next/link';
import Image from 'next/image';
import { useRestaurantInfo } from '@/hooks/use-restaurant-info';
import { useToast } from '@/hooks/use-toast';
import type { CarouselImage } from '@/lib/types';
import { Button } from '@/components/ui/button';
import {
  Card,
  CardContent,
  CardDescription,
  CardHeader,
  CardTitle,
} from '@/components/ui/card';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Textarea } from '@/components/ui/textarea';
import { ArrowLeft, Settings, Palette, Webhook, Type, Code, CreditCard, MessageCircle, FileText, UtensilsCrossed, Image as ImageIcon } from 'lucide-react';
import AdminNav from '@/components/admin-nav';
import { Separator } from '@/components/ui/separator';

// --- Funções de conversão de cor ---
function hexToHsl(hex: string): { h: number; s: number; l: number } | null {
  if (!hex) return null;
  const result = /^#?([a-f\d]{2})([a-f\d]{2})([a-f\d]{2})$/i.exec(hex);
  if (!result) return null;

  let r = parseInt(result[1], 16) / 255;
  let g = parseInt(result[2], 16) / 255;
  let b = parseInt(result[3], 16) / 255;

  const max = Math.max(r, g, b), min = Math.min(r, g, b);
  let h = 0, s = 0, l = (max + min) / 2;

  if (max === min) {
    h = s = 0; // achromatic
  } else {
    const d = max - min;
    s = l > 0.5 ? d / (2 - max - min) : d / (max + min);
    switch (max) {
      case r: h = (g - b) / d + (g < b ? 6 : 0); break;
      case g: h = (b - r) / d + 2; break;
      case b: h = (r - g) / d + 4; break;
    }
    h /= 6;
  }
  return { h: Math.round(h * 360), s: Math.round(s * 100), l: Math.round(l * 100) };
}

function hslToHex(h: number, s: number, l: number): string {
    s /= 100;
    l /= 100;

    let c = (1 - Math.abs(2 * l - 1)) * s,
        x = c * (1 - Math.abs((h / 60) % 2 - 1)),
        m = l - c/2,
        r = 0,
        g = 0,
        b = 0;

    if (0 <= h && h < 60) {
        r = c; g = x; b = 0;
    } else if (60 <= h && h < 120) {
        r = x; g = c; b = 0;
    } else if (120 <= h && h < 180) {
        r = 0; g = c; b = x;
    } else if (180 <= h && h < 240) {
        r = 0; g = x; b = c;
    } else if (240 <= h && h < 300) {
        r = x; g = 0; b = c;
    } else if (300 <= h && h < 360) {
        r = c; g = 0; b = x;
    }
    
    let toHex = (c: number) => {
        const hex = Math.round(c * 255).toString(16);
        return hex.length == 1 ? "0" + hex : hex;
    };

    return `#${toHex(r + m)}${toHex(g + m)}${toHex(b + m)}`;
}

function hslStringToHex(hslStr: string): string {
  if (!hslStr) return '#000000';
  const match = hslStr.match(/(\d+)\s*(\d+)%?\s*(\d+)%?/);
  if (!match) return '#000000';
  const [h, s, l] = match.slice(1).map(Number);
  if (isNaN(h) || isNaN(s) || isNaN(l)) return '#000000';
  return hslToHex(h, s, l);
}


export default function AdminSettingsPage() {
  const { restaurantInfo, updateRestaurantInfo, isLoaded } = useRestaurantInfo();
  const [name, setName] = useState('');
  const [logoUrl, setLogoUrl] = useState('');
  const [whatsappNumber, setWhatsappNumber] = useState('');
  const [footerText, setFooterText] = useState('');
  const [carouselImages, setCarouselImages] = useState<CarouselImage[]>([]);
  const [webhookUrl, setWebhookUrl] = useState('');
  const [facebookPixelHead, setFacebookPixelHead] = useState('');
  const [facebookPixelBody, setFacebookPixelBody] = useState('');
  const [googleTagManagerHead, setGoogleTagManagerHead] = useState('');
  const [googleTagManagerBody, setGoogleTagManagerBody] = useState('');
  const [googleAnalyticsHead, setGoogleAnalyticsHead] = useState('');
  const [googleAnalyticsBody, setGoogleAnalyticsBody] = useState('');
  const [mercadoPagoPublicKey, setMercadoPagoPublicKey] = useState('');
  const [colors, setColors] = useState({
    primary: '',
    background: '',
    accent: '',
    card: '',
    foreground: '',
    cardForeground: '',
    mutedForeground: ''
  });

  const { toast } = useToast();

  useEffect(() => {
    if (isLoaded) {
      setName(restaurantInfo.name);
      setLogoUrl(restaurantInfo.logoUrl);
      setWhatsappNumber(restaurantInfo.whatsappNumber);
      setFooterText(restaurantInfo.footerText);
      setCarouselImages(restaurantInfo.carouselImages);
      setWebhookUrl(restaurantInfo.webhookUrl || '');
      setFacebookPixelHead(restaurantInfo.facebookPixelHead || '');
      setFacebookPixelBody(restaurantInfo.facebookPixelBody || '');
      setGoogleTagManagerHead(restaurantInfo.googleTagManagerHead || '');
      setGoogleTagManagerBody(restaurantInfo.googleTagManagerBody || '');
      setGoogleAnalyticsHead(restaurantInfo.googleAnalyticsHead || '');
      setGoogleAnalyticsBody(restaurantInfo.googleAnalyticsBody || '');
      setMercadoPagoPublicKey(restaurantInfo.mercadoPagoPublicKey || '');
      setColors(restaurantInfo.colors);
    }
  }, [isLoaded, restaurantInfo]);
  
  const handleColorChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    const { name, value } = e.target;
    const hslColor = hexToHsl(value);
    if(hslColor) {
      setColors(prev => ({...prev, [name]: `${hslColor.h} ${hslColor.s}% ${hslColor.l}%`}));
    }
  }

  const handleCarouselImageChange = (index: number, value: string) => {
    const newImages = [...carouselImages];
    newImages[index].imageUrl = value;
    setCarouselImages(newImages);
  };


  const handleSubmit = (e: React.FormEvent) => {
    e.preventDefault();
    updateRestaurantInfo({
      name,
      logoUrl, 
      whatsappNumber,
      footerText,
      carouselImages,
      colors, 
      webhookUrl, 
      facebookPixelHead, 
      facebookPixelBody,
      googleTagManagerHead,
      googleTagManagerBody,
      googleAnalyticsHead,
      googleAnalyticsBody,
      mercadoPagoPublicKey
    });
    toast({
      title: 'Sucesso!',
      description: 'As configurações do restaurante foram atualizadas.',
    });
  };

  const ColorInput = ({ label, name, value, description }: { label: string, name: keyof typeof colors, value: string, description: string }) => (
    <div className="space-y-2">
      <Label htmlFor={name}>{label}</Label>
      <div className="flex items-center gap-4">
        <Input
          id={name}
          name={name}
          type="color"
          value={hslStringToHex(value)}
          onChange={handleColorChange}
          className="h-12 w-12 p-1"
        />
        <div className="flex-1">
          <Input 
            value={value} 
            onChange={(e) => setColors(prev => ({...prev, [name]: e.target.value}))}
            placeholder="Ex: 20 100% 66%"
          />
          <p className="text-sm text-muted-foreground mt-1">
            {description}
          </p>
        </div>
      </div>
    </div>
  );

  return (
    <div className="max-w-7xl mx-auto p-4 md:p-6">
      <div className="flex items-center gap-4 mb-6">
        <Link href="/">
          <Button variant="outline" size="icon">
            <ArrowLeft className="h-4 w-4" />
          </Button>
        </Link>
        <h1 className="text-3xl font-bold">Configurações</h1>
      </div>

      <AdminNav />

      <form onSubmit={handleSubmit}>
        <Card className="mb-8">
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <Settings className="h-6 w-6" />
              Configurações Gerais
            </CardTitle>
            <CardDescription>
              Ajuste as informações e a aparência do seu cardápio.
            </CardDescription>
          </CardHeader>
          <CardContent className="space-y-6">

             <div className="space-y-2">
              <Label htmlFor="name" className="flex items-center gap-2"><UtensilsCrossed className="h-4 w-4" /> Nome do Cardápio</Label>
              <Input
                id="name"
                name="name"
                value={name}
                onChange={(e) => setName(e.target.value)}
                placeholder="Nome do seu restaurante"
                required
              />
              <p className="text-sm text-muted-foreground">
                Este nome aparecerá no topo do seu cardápio.
              </p>
            </div>

            <Separator />

            <div className="space-y-2">
              <Label htmlFor="logoUrl">URL da Logo</Label>
              <Input
                id="logoUrl"
                name="logoUrl"
                value={logoUrl}
                onChange={(e) => setLogoUrl(e.target.value)}
                placeholder="https://exemplo.com/sua-logo.png"
                required
              />
              <p className="text-sm text-muted-foreground">
                Cole a URL de uma imagem para ser a logo do seu restaurante.
              </p>
            </div>

            {logoUrl && (
              <div className="space-y-2">
                <Label>Pré-visualização da Logo</Label>
                <div className="w-24 h-24 relative border rounded-md flex items-center justify-center bg-muted">
                  <Image
                    src={logoUrl}
                    alt="Pré-visualização da Logo"
                    fill
                    className="object-contain p-2"
                    onError={() => {
                        toast({
                            variant: 'destructive',
                            title: 'Erro ao carregar imagem',
                            description: 'Não foi possível carregar a imagem da URL fornecida.'
                        })
                    }}
                  />
                </div>
              </div>
            )}
            
            <Separator />
            
            <div className="space-y-2">
                <Label htmlFor="whatsappNumber" className="flex items-center gap-2"><MessageCircle className="h-4 w-4" /> Número do WhatsApp</Label>
                <Input
                    id="whatsappNumber"
                    name="whatsappNumber"
                    value={whatsappNumber}
                    onChange={(e) => setWhatsappNumber(e.target.value)}
                    placeholder="5511999998888"
                    required
                />
                <p className="text-sm text-muted-foreground">
                    Insira o número que receberá os pedidos, incluindo o código do país (Ex: 55 para Brasil).
                </p>
            </div>

            <Separator />

            <div className="space-y-2">
                <Label htmlFor="footerText" className="flex items-center gap-2"><FileText className="h-4 w-4" /> Texto do Rodapé</Label>
                <Input
                    id="footerText"
                    name="footerText"
                    value={footerText}
                    onChange={(e) => setFooterText(e.target.value)}
                    placeholder="Todos os direitos reservados."
                />
                <p className="text-sm text-muted-foreground">
                    Este texto aparecerá no rodapé, após o nome do restaurante e o ano.
                </p>
            </div>

          </CardContent>
        </Card>

        <Card className="mb-8">
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <ImageIcon className="h-6 w-6" />
              Imagens do Carrossel
            </CardTitle>
            <CardDescription>
              Altere as imagens que aparecem no slider da página inicial. Use uma proporção de 1200x500 pixels.
            </CardDescription>
          </CardHeader>
          <CardContent className="space-y-6">
            {carouselImages.map((image, index) => (
              <div key={image.id} className="space-y-2">
                <Label htmlFor={`carousel-image-${index}`}>Imagem ${index + 1}</Label>
                <div className="flex gap-4 items-start">
                   {image.imageUrl && (
                     <div className="w-32 h-20 relative border rounded-md flex items-center justify-center bg-muted shrink-0">
                        <Image
                            src={image.imageUrl}
                            alt={`Preview ${index + 1}`}
                            fill
                            className="object-cover rounded-md"
                        />
                     </div>
                   )}
                   <div className="flex-grow">
                     <Input
                        id={`carousel-image-${index}`}
                        value={image.imageUrl}
                        onChange={(e) => handleCarouselImageChange(index, e.target.value)}
                        placeholder="https://.../imagem-do-slider.png"
                      />
                      <p className="text-xs text-muted-foreground mt-1">{image.description}</p>
                   </div>
                </div>
              </div>
            ))}
          </CardContent>
        </Card>

        <Card className="mb-8">
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <Palette className="h-6 w-6" />
              Cores do Cardápio
            </CardTitle>
            <CardDescription>
              Personalize as cores do seu cardápio. Você pode usar o seletor visual ou inserir valores HSL (Ex: 20 100% 66%).
            </CardDescription>
          </CardHeader>
          <CardContent className="space-y-6">
            <ColorInput
              label="Cor Primária"
              name="primary"
              value={colors.primary}
              description="Usada em botões e links principais."
            />
             <ColorInput
              label="Cor de Fundo"
              name="background"
              value={colors.background}
              description="A cor principal do fundo do seu cardápio."
            />
             <ColorInput
              label="Fundo do Card"
              name="card"
              value={colors.card}
              description="A cor de fundo para os cards de produtos."
            />
             <ColorInput
              label="Cor de Destaque"
              name="accent"
              value={colors.accent}
              description="Usada para notificações e detalhes visuais."
            />
          </CardContent>
        </Card>
        
        <Card className="mb-8">
           <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <Type className="h-6 w-6" />
              Cores de Texto
            </CardTitle>
            <CardDescription>
              Ajuste as cores dos textos para garantir a legibilidade.
            </CardDescription>
          </CardHeader>
          <CardContent className="space-y-6">
            <ColorInput
              label="Títulos e Textos Principais"
              name="foreground"
              value={colors.foreground}
              description="Cor para títulos e textos importantes."
            />
             <ColorInput
              label="Textos em Cards"
              name="cardForeground"
              value={colors.cardForeground}
              description="Cor do texto que aparece dentro de cards e caixas."
            />
             <ColorInput
              label="Descrições e Textos Secundários"
              name="mutedForeground"
              value={colors.mutedForeground}
              description="Cor para descrições de produtos e textos menos importantes."
            />
          </CardContent>
        </Card>

        <Card className="mb-8">
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <CreditCard className="h-6 w-6" />
              Pagamentos (Mercado Pago)
            </CardTitle>
            <CardDescription>
              Configure a integração com o Mercado Pago para aceitar pagamentos online.
            </CardDescription>
          </CardHeader>
          <CardContent>
            <div className="space-y-2">
              <Label htmlFor="mercadoPagoPublicKey">Public Key</Label>
              <Input
                id="mercadoPagoPublicKey"
                name="mercadoPagoPublicKey"
                value={mercadoPagoPublicKey}
                onChange={(e) => setMercadoPagoPublicKey(e.target.value)}
                placeholder="APP_USR-..."
              />
              <p className="text-sm text-muted-foreground">
                Sua chave pública do Mercado Pago. Encontrada nas suas credenciais de produção.
              </p>
            </div>
          </CardContent>
        </Card>

        <Card className="mb-8">
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <Webhook className="h-6 w-6" />
              Automações (Webhook)
            </CardTitle>
            <CardDescription>
              Insira uma URL de webhook para receber os detalhes dos pedidos e criar automações.
            </CardDescription>
          </CardHeader>
          <CardContent>
            <div className="space-y-2">
              <Label htmlFor="webhookUrl">URL do Webhook</Label>
              <Input
                id="webhookUrl"
                name="webhookUrl"
                value={webhookUrl}
                onChange={(e) => setWebhookUrl(e.target.value)}
                placeholder="https://sua-automacao.com/webhook"
              />
              <p className="text-sm text-muted-foreground">
                A cada novo pedido, enviaremos os dados para esta URL via POST.
              </p>
            </div>
          </CardContent>
        </Card>
        
        <Card>
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <Code className="h-6 w-6" />
              Marketing e Rastreamento
            </CardTitle>
            <CardDescription>
              Adicione scripts de rastreamento como Pixel do Facebook, Google Analytics e Google Tag Manager.
            </CardDescription>
          </CardHeader>
          <CardContent className="space-y-6">
            <div className="space-y-4 p-4 border rounded-md">
                <h3 className="font-medium text-lg">Pixel do Facebook</h3>
                <div className="space-y-2">
                  <Label htmlFor="facebookPixelHead">Código Pixel (Head)</Label>
                  <Textarea
                    id="facebookPixelHead"
                    name="facebookPixelHead"
                    value={facebookPixelHead}
                    onChange={(e) => setFacebookPixelHead(e.target.value)}
                    placeholder="Cole aqui o código do Pixel para a tag <head>"
                    rows={5}
                  />
                  <p className="text-sm text-muted-foreground">
                    Este código será inserido no final da tag &lt;head&gt; em todas as páginas.
                  </p>
                </div>
                <div className="space-y-2">
                  <Label htmlFor="facebookPixelBody">Script de Rastreamento (Body)</Label>
                  <Textarea
                    id="facebookPixelBody"
                    name="facebookPixelBody"
                    value={facebookPixelBody}
                    onChange={(e) => setFacebookPixelBody(e.target.value)}
                    placeholder="Cole aqui o código <noscript> para o início da tag <body>"
                    rows={5}
                  />
                  <p className="text-sm text-muted-foreground">
                    Este código será inserido no início da tag &lt;body&gt;.
                  </p>
                </div>
            </div>
            <Separator />
            <div className="space-y-4 p-4 border rounded-md">
                <h3 className="font-medium text-lg">Google Tag Manager</h3>
                 <div className="space-y-2">
                  <Label htmlFor="googleTagManagerHead">Google Tag Manager (Head)</Label>
                  <Textarea
                    id="googleTagManagerHead"
                    name="googleTagManagerHead"
                    value={googleTagManagerHead}
                    onChange={(e) => setGoogleTagManagerHead(e.target.value)}
                    placeholder="Cole aqui o código do GTM para a tag <head>"
                    rows={5}
                  />
                   <p className="text-sm text-muted-foreground">
                    Este código será inserido no final da tag &lt;head&gt;.
                  </p>
                </div>
                <div className="space-y-2">
                    <Label htmlFor="googleTagManagerBody">Google Tag Manager (Body)</Label>
                    <Textarea
                        id="googleTagManagerBody"
                        name="googleTagManagerBody"
                        value={googleTagManagerBody}
                        onChange={(e) => setGoogleTagManagerBody(e.target.value)}
                        placeholder="Cole aqui o código <noscript> do GTM para o início da tag <body>"
                        rows={5}
                    />
                    <p className="text-sm text-muted-foreground">
                        Este código será inserido no início da tag &lt;body&gt;.
                    </p>
                </div>
            </div>
            <Separator />
            <div className="space-y-4 p-4 border rounded-md">
                <h3 className="font-medium text-lg">Google Analytics</h3>
                 <div className="space-y-2">
                  <Label htmlFor="googleAnalyticsHead">Google Analytics (Head)</Label>
                  <Textarea
                    id="googleAnalyticsHead"
                    name="googleAnalyticsHead"
                    value={googleAnalyticsHead}
                    onChange={(e) => setGoogleAnalyticsHead(e.target.value)}
                    placeholder="Cole aqui o script do GA4 para a tag <head>"
                    rows={5}
                  />
                  <p className="text-sm text-muted-foreground">
                    Este código será inserido no final da tag &lt;head&gt;.
                  </p>
                </div>
                 <div className="space-y-2">
                  <Label htmlFor="googleAnalyticsBody">Google Analytics (Body)</Label>
                  <Textarea
                    id="googleAnalyticsBody"
                    name="googleAnalyticsBody"
                    value={googleAnalyticsBody}
                    onChange={(e) => setGoogleAnalyticsBody(e.target.value)}
                    placeholder="Cole aqui um código relacionado ao GA4 para o início da tag <body> (se houver)"
                    rows={5}
                  />
                  <p className="text-sm text-muted-foreground">
                    Este código será inserido no início da tag &lt;body&gt;.
                  </p>
                </div>
            </div>
          </CardContent>
        </Card>

        <div className="mt-8">
            <Button type="submit">Salvar Todas as Alterações</Button>
        </div>
      </form>
    </div>
  );
}
